<?php
/**
 * Main plugin class file.
 *
 * @package WordPress Plugin Template/Includes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main plugin class.
 */
class Supreme_Mega_Menu {

	/**
	 * The single instance of Supreme_Mega_Menu.
	 *
	 * @var     object
	 * @access  private
	 * @since   1.0.0
	 */
	private static $_instance = null; //phpcs:ignore

	/**
	 * Local instance of Supreme_Mega_Menu_Admin_API
	 *
	 * @var Supreme_Mega_Menu_Admin_API|null
	 */
	public $admin = null;

	/**
	 * Settings class object
	 *
	 * @var     object
	 * @access  public
	 * @since   1.0.0
	 */
	public $settings = null;

	/**
	 * The version number.
	 *
	 * @var     string
	 * @access  public
	 * @since   1.0.0
	 */
	public $_version; //phpcs:ignore

	/**
	 * The token.
	 *
	 * @var     string
	 * @access  public
	 * @since   1.0.0
	 */
	public $_token; //phpcs:ignore

	/**
	 * The main plugin file.
	 *
	 * @var     string
	 * @access  public
	 * @since   1.0.0
	 */
	public $file;

	/**
	 * The main plugin directory.
	 *
	 * @var     string
	 * @access  public
	 * @since   1.0.0
	 */
	public $dir;

	/**
	 * The plugin assets directory.
	 *
	 * @var     string
	 * @access  public
	 * @since   1.0.0
	 */
	public $assets_dir;

	/**
	 * The plugin assets URL.
	 *
	 * @var     string
	 * @access  public
	 * @since   1.0.0
	 */
	public $assets_url;

	/**
	 * Suffix for JavaScripts.
	 *
	 * @var     string
	 * @access  public
	 * @since   1.0.0
	 */
	public $script_suffix;

	/**
	 * Constructor funtion.
	 *
	 * @param string $file File constructor.
	 * @param string $version Plugin version.
	 */
	public function __construct( $file = '', $version = '1.0.0' ) {
		$this->_version = $version;
		$this->_token   = 'supreme_mega_menu';

		// Load plugin environment variables.
		$this->file       = $file;
		$this->dir        = dirname( $this->file );
		$this->assets_dir = trailingslashit( $this->dir ) . 'assets';
		$this->assets_url = esc_url( trailingslashit( plugins_url( '/assets/', $this->file ) ) );

		$this->script_suffix = '.min';

		register_activation_hook( $this->file, array( $this, 'install' ) );
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/persist-admin-notices-dismissal/persist-admin-notices-dismissal.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/license/class.license-load.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/license/class.updater.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/license/class.licence.php';
		// Load frontend JS & CSS.
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_styles' ), 20 );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ), 20 );

		// Load admin JS & CSS.
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ), 10, 1 );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_styles' ), 10, 1 );

		// Load API for generic admin functions.
		if ( is_admin() ) {
			$this->admin = new Supreme_Mega_Menu_Admin_API();
		}

		// Handle localisation.
		$this->load_plugin_textdomain();
		add_action( 'init', array( $this, 'load_localisation' ), 0 );

		// Load Divi.
		add_action( 'divi_extensions_init', array( $this, 'supreme_mega_menu_initialize_extension' ) );
	} // End __construct ()

	/**
	 * Register post type function.
	 *
	 * @param string $post_type Post Type.
	 * @param string $plural Plural Label.
	 * @param string $single Single Label.
	 * @param string $description Description.
	 * @param array  $options Options array.
	 *
	 * @return bool|string|Supreme_Mega_Menu_Post_Type
	 */
	public function register_post_type( $post_type = '', $plural = '', $single = '', $description = '', $options = array() ) {

		if ( ! $post_type || ! $plural || ! $single ) {
			return false;
		}

		$post_type = new Supreme_Mega_Menu_Post_Type( $post_type, $plural, $single, $description, $options );

		return $post_type;
	}

	/**
	 * Wrapper function to register a new taxonomy.
	 *
	 * @param string $taxonomy Taxonomy.
	 * @param string $plural Plural Label.
	 * @param string $single Single Label.
	 * @param array  $post_types Post types to register this taxonomy for.
	 * @param array  $taxonomy_args Taxonomy arguments.
	 *
	 * @return bool|string|Supreme_Mega_Menu_Taxonomy
	 */
	public function register_taxonomy( $taxonomy = '', $plural = '', $single = '', $post_types = array(), $taxonomy_args = array() ) {

		if ( ! $taxonomy || ! $plural || ! $single ) {
			return false;
		}

		$taxonomy = new Supreme_Mega_Menu_Taxonomy( $taxonomy, $plural, $single, $post_types, $taxonomy_args );

		return $taxonomy;
	}

	/**
	 * Load frontend CSS.
	 *
	 * @access  public
	 * @return void
	 * @since   1.0.0
	 */
	public function enqueue_styles() {
		if ( 'on' === get_option( 'smega_menu_dynamic_css' ) ) {
			if ( function_exists( 'et_core_is_fb_enabled' ) ) {
				if ( ! et_core_is_fb_enabled() ) {
					wp_dequeue_style( 'supreme-mega-menu-styles' );
				} else {
					wp_enqueue_style( $this->_token . '-builder', esc_url( $this->assets_url ) . 'css/supreme-mega-menu-builder.css', array(), $this->_version );
				}
			}
		} else {
			wp_enqueue_style( $this->_token . '-styles', esc_url( trailingslashit( plugins_url( '/styles/', $this->file ) ) ) . 'style.min.css', array(), $this->_version );
		}

	} // End enqueue_styles ()

	/**
	 * Load frontend Javascript.
	 *
	 * @access  public
	 * @return  void
	 * @since   1.0.0
	 */
	public function enqueue_scripts() {
		wp_dequeue_script( 'supreme-mega-menu-frontend-bundle' );
	} // End enqueue_scripts ()

	/**
	 * Admin enqueue style.
	 *
	 * @param string $hook Hook parameter.
	 *
	 * @return void
	 */
	public function admin_enqueue_styles( $hook = '' ) {
		wp_register_style( $this->_token . '-admin', esc_url( $this->assets_url ) . 'css/admin.css', array(), $this->_version );
		// wp_enqueue_style( $this->_token . '-admin' ); .
	} // End admin_enqueue_styles ()

	/**
	 * Load admin Javascript.
	 *
	 * @access  public
	 *
	 * @param string $hook Hook parameter.
	 *
	 * @return  void
	 * @since   1.0.0
	 */
	public function admin_enqueue_scripts( $hook = '' ) {
		wp_register_script( $this->_token . '-admin', esc_url( $this->assets_url ) . 'js/admin' . $this->script_suffix . '.js', array( 'jquery' ), $this->_version, true );
		// wp_enqueue_script( $this->_token . '-admin' );.
	} // End admin_enqueue_scripts ()

	/**
	 * Load plugin localisation
	 *
	 * @access  public
	 * @return  void
	 * @since   1.0.0
	 */
	public function load_localisation() {
		load_plugin_textdomain( 'supreme-mega-menu', false, dirname( plugin_basename( $this->file ) ) . '/lang/' );
	} // End load_localisation ()

	/**
	 * Load plugin textdomain
	 *
	 * @access  public
	 * @return  void
	 * @since   1.0.0
	 */
	public function load_plugin_textdomain() {
		$domain = 'supreme-mega-menu';

		$locale = apply_filters( 'plugin_locale', get_locale(), $domain );

		load_textdomain( $domain, WP_LANG_DIR . '/' . $domain . '/' . $domain . '-' . $locale . '.mo' );
		load_plugin_textdomain( $domain, false, dirname( plugin_basename( $this->file ) ) . '/lang/' );
	} // End load_plugin_textdomain ()

	/**
	 * Main Supreme_Mega_Menu Instance
	 *
	 * Ensures only one instance of Supreme_Mega_Menu is loaded or can be loaded.
	 *
	 * @param string $file File instance.
	 * @param string $version Version parameter.
	 *
	 * @return Object Supreme_Mega_Menu instance
	 * @see Supreme_Mega_Menu()
	 * @since 1.0.0
	 * @static
	 */
	public static function instance( $file = '', $version = '1.0.0' ) {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self( $file, $version );
		}

		return self::$_instance;
	} // End instance ()

	/**
	 * Cloning is forbidden.
	 *
	 * @since 1.0.0
	 */
	public function __clone() {
		_doing_it_wrong( __FUNCTION__, esc_html( __( 'Cloning of Supreme_Mega_Menu is forbidden' ) ), esc_attr( $this->_version ) );

	} // End __clone ()

	/**
	 * Unserializing instances of this class is forbidden.
	 *
	 * @since 1.0.0
	 */
	public function __wakeup() {
		_doing_it_wrong( __FUNCTION__, esc_html( __( 'Unserializing instances of Supreme_Mega_Menu is forbidden' ) ), esc_attr( $this->_version ) );
	} // End __wakeup ()

	/**
	 * Installation. Runs on activation.
	 *
	 * @access  public
	 * @return  void
	 * @since   1.0.0
	 */
	public function install() {
		$this->_log_version_number();
	} // End install ()

	/**
	 * Log the plugin version number.
	 *
	 * @access  public
	 * @return  void
	 * @since   1.0.0
	 */
	private function _log_version_number() { //phpcs:ignore
		update_option( $this->_token . '_version', $this->_version );
	} // End _log_version_number ()

	/**
	 * Initialize Supreme Mega Menu.
	 *
	 * @access  public
	 * @return  void
	 * @since   1.0.0
	 */
	public function supreme_mega_menu_initialize_extension() {
		require_once plugin_dir_path( __FILE__ ) . 'MegaMenu.php';

		add_action(
			'wp_ajax_mega_menu_get_menu_object',
			function () {
				function recursive_merge_child_menu( $menu_items, $menu ) {
					$found_child = array();

					foreach ( $menu_items as $curr_menu_item ) {
						if ( (int) $curr_menu_item->menu_item_parent === $menu->ID ) {
							$found_child[] = array(
								'title'    => $curr_menu_item->post_title,
								'link'     => $curr_menu_item->url,
								'children' => recursive_merge_child_menu( $menu_items, $curr_menu_item ),
							);
						}
					}

					return $found_child;
				}

				function organize_menu_items( $menu_items ) {
					$final_menu_items = array();

					foreach ( $menu_items as $menu_item ) {
						$is_sub_item = '0' !== $menu_item->menu_item_parent;
						$title       = $menu_item->post_title;
						$link        = $menu_item->url;
						$menu_id     = $menu_item->ID;

						if ( $is_sub_item ) {
							continue;
						}

						$final_menu_items[] = array(
							'title'    => $title,
							'link'     => $link,
							'children' => recursive_merge_child_menu( $menu_items, $menu_item ),
						);
					}

					return $final_menu_items;
				}

				$menu_id = isset( $_REQUEST['id'] ) ? $_REQUEST['id'] : null;

				$required_menu = wp_get_nav_menu_items( $menu_id );
				echo wp_json_encode( organize_menu_items( $required_menu ) );

				wp_die();
			}
		);

		add_filter(
			'woocommerce_add_to_cart_fragments',
			function( $fragments ) {
				$fragments['.dsm-mega-menu-woo-cart-count'] = sprintf( '<span class="dsm-mega-menu-woo-cart-count">%1$s</span>', WC()->cart->get_cart_contents_count() );
				return $fragments;
			}
		);
	}
}
